/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: OS_SPICustom.c 12222 2018-07-26 07:03:23Z Robert $:

  Description:
    Implementation of the custom SPI abstration layer

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2018-07-26  Added return value to OS_SpiInit() and changed compile #error to
                #warning
    2014-08-27  created

**************************************************************************************/

/*****************************************************************************/
/*! \file OS_SPICustom.c
*    Sample SPI abstraction layer. Implementation must be done
*    according to used target system                                         */
/*****************************************************************************/

#include "OS_Spi.h"
#include "ToolkitSample.h"
#include <hal_resources_defines.h>

#ifdef CIFX_TOOLKIT_HWIF
  #warning "Implement SPI target system abstraction in this file"
#endif

/*****************************************************************************/
/*!  \addtogroup CIFX_TK_OS_ABSTRACTION Operating System Abstraction
*    \{                                                                      */
/*****************************************************************************/
#define SERDPM_SLAVE 1
static NX500_SPI_AREA_T* s_ptSpi = (NX500_SPI_AREA_T*) NX500_NETX_SPI_AREA;

/*****************************************************************************/
/*! Initialize SPI components
*   \param pvOSDependent OS Dependent parameter
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
long OS_SpiInit(void* pvOSDependent)
{
  /* initialize SPI device */

  /* soft reset spi and clear both fifos */
  s_ptSpi->ulSpi_control_register = MSK_NX500_spi_control_register_CR_softreset  |
                                    MSK_NX500_spi_control_register_CR_clr_infifo |
                                    MSK_NX500_spi_control_register_CR_clr_outfifo;

  /* setup SPI Core data width and enable static nCS handling */
  s_ptSpi->ulSpi_control_register  = 7 << SRT_NX500_spi_control_register_CR_burst;

  /* set clock to 12,5MHz*/
  s_ptSpi->ulSpi_control_register |= (12 << SRT_NX500_spi_control_register_CR_speed);

  /* SPI Mode 3 */
  s_ptSpi->ulSpi_control_register |= MSK_NX500_spi_control_register_CR_cpol;

  s_ptSpi->ulSpi_control_register |= MSK_NX500_spi_control_register_CR_ms |
                                     MSK_NX500_spi_control_register_CR_en |
                                     MSK_NX500_spi_control_register_CR_read |
                                     MSK_NX500_spi_control_register_CR_write;

  /* reset status bits */
  s_ptSpi->ulSpi_status_register = 0;

  /* do not use irqs for now */
  s_ptSpi->ulSpi_interrupt_control_register = 0;

  return 0;
}

/*****************************************************************************/
/*! Assert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiAssert(void* pvOSDependent)
{
  /* Clear tx/rx FIFO */
  s_ptSpi->ulSpi_control_register |= (MSK_NX500_spi_control_register_CR_clr_infifo | MSK_NX500_spi_control_register_CR_clr_outfifo );
  /* assert chip select */
  s_ptSpi->ulSpi_control_register |= (1 << (SRT_NX500_spi_control_register_CR_ss+SERDPM_SLAVE));
}

/*****************************************************************************/
/*! Deassert chip select
*   \param pvOSDependent OS Dependent parameter to identify card             */
/*****************************************************************************/
void OS_SpiDeassert(void* pvOSDependent)
{
  /* deassert chip select */
  s_ptSpi->ulSpi_control_register &= ~(1 << (SRT_NX500_spi_control_register_CR_ss+SERDPM_SLAVE));
}

/*****************************************************************************/
/*! Lock the SPI bus
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiLock(void* pvOSDependent)
{
  /* lock access to SPI device */
  NONOS_DEVICEINSTANCE_T* ptInst = (NONOS_DEVICEINSTANCE_T*)pvOSDependent;
  lock_irqfiq_save(ptInst->ulSerLock);  /*lint !e522 !e529 !e160 : function call lock_irqfiq_save() */
} /*lint !e438 !e529 */

/*****************************************************************************/
/*! Unlock the SPI bus
*   \param pvOSDependent OS Dependent parameter                              */
/*****************************************************************************/
void OS_SpiUnlock(void* pvOSDependent)
{
  /* unlock access to SPI device */
  NONOS_DEVICEINSTANCE_T* ptInst = (NONOS_DEVICEINSTANCE_T*)pvOSDependent;
  lock_irqfiq_restore(ptInst->ulSerLock); /*lint !e522 !e529 !e160 : function call lock_irqfiq_restore() */
} /*lint !e438 !e529 */

/*****************************************************************************/
/*! Transfer byte stream via SPI
*   \param pvOSDependent OS Dependent parameter to identify card
*   \param pbSend        Send buffer (NULL for polling)
*   \param pbRecv        Receive buffer (NULL if discard)
*   \param ulLen         Length of SPI transfer                              */
/*****************************************************************************/
void OS_SpiTransfer(void* pvOSDependent, uint8_t* pbSend, uint8_t* pbRecv, uint32_t ulLen)
{
  uint32_t ulOutFuel;

  while (ulLen--)
  {
    /* write byte to spi bus */
    if (pbSend == NULL)
      s_ptSpi->ulSpi_data_register = MSK_NX500_spi_data_register_dr_valid0;
    else
      s_ptSpi->ulSpi_data_register = *(pbSend++) | MSK_NX500_spi_data_register_dr_valid0;

    /* wait until all bytes are clocked out. There will be a byte in the
     * receive buffer.  */
    do
    {
      ulOutFuel   = s_ptSpi->ulSpi_status_register & MSK_NX500_spi_status_register_SR_out_fuel_val;
    } while( ulOutFuel!=0 );

    /* get the received byte */
    if (pbRecv == NULL)
      s_ptSpi->ulSpi_data_register;
    else
      *(pbRecv++) = s_ptSpi->ulSpi_data_register & MSK_NX500_spi_data_register_data_byte_0;
  }
}
/*****************************************************************************/
/*! \}                                                                       */
/*****************************************************************************/
